/* Copyright (C) 2013-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef UUID_2cee7b5a_42ec_11e3_a170_001cc047a813
#define UUID_2cee7b5a_42ec_11e3_a170_001cc047a813

#include "vncint.h"

/**
 * \file vncmirrorlinktypes.h
 *
 * This file defines common MirrorLink data types used by VNC Mobile Solution
 * components.
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the applicationCategory field in the
 * VNCContextInformation structure.
 * \else
 * \brief Constants used to define MirrorLink application categories and
 * subcategories
 * \endif
 *
 * \if VIEWER_SDK
 * The applicationCategory field in the VNCContextInformation places the
 * reported server application into one of a number of pre-defined categories
 * and sub-categories.  This allows a viewer application sensibly to implement
 * a driver distraction policy without knowledge of every possible server
 * application.
 * \endif
 *
 * \if VIEWER_SDK
 * Use VNCApplicationCategoryCategoryMASK if the driver distraction policy is
 * based only on the category of the application, and not on the
 * sub-category:
 *
 * \code
 * VNCContextInformation contextInformation;
 * ...
 * if ((contextInformation.applicationCategory & VNCApplicationCategoryCategoryMASK) ==
 *     VNCApplicationCategoryMessaging)
 * {
 *     // It's some kind of messaging application - we don't care exactly which
 * }
 * \endcode
 * \else
 * Use VNCApplicationCategoryCategoryMASK to mask out the category from the
 * subcategory.
 * \endif
 *
 * The values of this enum which include subcategory, such as
 * VNCApplicationCategoryUIHomeScreen already include the category, such as
 * VNCApplicationCategoryUI, so there is no need to perform this masking if
 * values containing subcategories are used.
 *
 * For further information, see Appendix A, 'UPnP Application Server Service'
 * in the MirrorLink specification.
 *
 * \if VIEWER_SDK
 * \see VNCContextInformation, VNCViewerContextInformationCallback
 * \endif
 */
typedef enum
{
    /** Used to mask out the category from the subcategory for testing. */
    VNCApplicationCategoryCategoryMASK                   = 0xFFFF0000,

    /** The server has no information about the application category. */
    VNCApplicationCategoryUnknown                        = 0x00000000,

    /** General UI framework category */
    VNCApplicationCategoryUI                             = 0x00010000,
    /** Home screen / start-up screen */
    VNCApplicationCategoryUIHomeScreen                   = 0x00010001,
    /** Menu */
    VNCApplicationCategoryUIMenu                         = 0x00010002,
    /** Notification */
    VNCApplicationCategoryUINotification                 = 0x00010003,
    /** Application listing */
    VNCApplicationCategoryUIApplicationListing           = 0x00010004,
    /** Settings */
    VNCApplicationCategoryUISettings                     = 0x00010005,
    /** Immersive home category */
    VNCApplicationCategoryUIImmersiveHomeScreen          = 0x00010006,

    /** General phone call application category */
    VNCApplicationCategoryPhone                          = 0x00020000,
    /** Contact list */
    VNCApplicationCategoryPhoneContactList               = 0x00020001,
    /** Call log */
    VNCApplicationCategoryPhoneCallLog                   = 0x00020002,
    /** Immersive phone */
    VNCApplicationCategoryPhoneImmersiveCall             = 0x00020003,

    /** General media applications category */
    VNCApplicationCategoryMedia                          = 0x00030000,
    /** Music */
    VNCApplicationCategoryMediaMusic                     = 0x00030001,
    /** Video */
    VNCApplicationCategoryMediaVideo                     = 0x00030002,
    /** Gaming */
    VNCApplicationCategoryMediaGaming                    = 0x00030003,
    /** Image */
    VNCApplicationCategoryMediaImage                     = 0x00030004,

    /** General messaging applications category */
    VNCApplicationCategoryMessaging                      = 0x00040000,
    /** SMS */
    VNCApplicationCategoryMessagingSMS                   = 0x00040001,
    /** MMS */
    VNCApplicationCategoryMessagingMMS                   = 0x00040002,
    /** Email */
    VNCApplicationCategoryMessagingEmail                 = 0x00040003,

    /** General navigation category */
    VNCApplicationCategoryNavigation                     = 0x00050000,

    /** General browser category */
    VNCApplicationCategoryBrowser                        = 0x00060000,
    /** Application store */
    VNCApplicationCategoryBrowserApplicationStore        = 0x00060001,

    /** General productivity category */
    VNCApplicationCategoryProductivity                   = 0x00070000,
    /** Document viewer */
    VNCApplicationCategoryProductivityDocumentViewer     = 0x00070001,
    /** Document editor */
    VNCApplicationCategoryProductivityDocumentEditor     = 0x00070002,

    /** General information category */
    VNCApplicationCategoryInformation                    = 0x00080000,
    /** News */
    VNCApplicationCategoryInformationNews                = 0x00080001,
    /** Weather */
    VNCApplicationCategoryInformationWeather             = 0x00080002,
    /** Stocks */
    VNCApplicationCategoryInformationStocks              = 0x00080003,
    /** Travel */
    VNCApplicationCategoryInformationTravel              = 0x00080004,
    /** Sports */
    VNCApplicationCategoryInformationSports              = 0x00080005,
    /** Clock */
    VNCApplicationCategoryInformationClock               = 0x00080006,

    /** General social networking category */
    VNCApplicationCategorySocialNetworking               = 0x00090000,

    /** General Personal Information Management category */
    VNCApplicationCategoryPIM                            = 0x000A0000,
    /** Calendar */
    VNCApplicationCategoryPIMCalendar                    = 0x000A0001,
    /** Notes */
    VNCApplicationCategoryPIMNotes                       = 0x000A0002,

    /** General UI-less applications category */
    VNCApplicationCategoryNoUI                           = 0xF0000000,
    /** Server functionality */
    VNCApplicationCategoryNoUIServer                     = 0xF0000001,
    /** Client functionality */
    VNCApplicationCategoryNoUIClient                     = 0xF0000002,
    /** Voice command engine */
    VNCApplicationCategoryNoUIVoiceCommandEngine         = 0xF0000010,
    /** Conversational audio. Used only in RTP header extensions */
    VNCApplicationCategoryNoUIConversationalAudio        = 0xF0000020,
    /** Category used within the context information to tell the client is
     * should switch to the native UI, or disconnect the VNC session */
    VNCApplicationCategorySwitchToClientNativeUI         = 0xF000FFFF,

    /** General testing and certification category */
    VNCApplicationCategoryTestingAndCertification        = 0xFFFE0000,

    /** General system category */
    VNCApplicationCategorySystem                         = 0xFFFF0000,
    /** PIN input for device unlock */
    VNCApplicationCategorySystemInputUnlockPIN           = 0xFFFF0001,
    /** Bluetooth PIN code input */
    VNCApplicationCategorySystemInputBluetoothPIN        = 0xFFFF0002,
    /** Other password input */
    VNCApplicationCategorySystemInputOtherPassword       = 0xFFFF000F,
    /** Voice command confirmation */
    VNCApplicationCategorySystemVoiceCommandConfirmation = 0xFFFF0010,

    /** \deprecated - Use VNCApplicationCategoryNoUIServer instead. */
    VNCApplicationCategoryNoUIAudioServer                = VNCApplicationCategoryNoUIServer,
    /** \deprecated - Use VNCApplicationCategoryNoUIClient instead. */
    VNCApplicationCategoryNoUIAudioClient                = VNCApplicationCategoryNoUIClient,

} VNCApplicationCategory;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the contentCategory field in the
 * VNCContextInformation structure.
 * \else
 * \brief Constants used to define MirrorLink visual content categories.
 * \endif
 *
 * This field is a bit mask that specifies the type or types of content that an
 * application will render into the server display.
 *
 * For further information, see Appendix A, 'UPnP Application Server Service'
 * in the MirrorLink specification.
 *
 * \if VIEWER_SDK
 * \see VNCContextInformation, VNCViewerContextInformationCallback
 * \endif
 */
typedef enum
{
    /** Text */
    VNCVisualContentCategoryText           = 0x00000001,
    /** Video */
    VNCVisualContentCategoryVideo          = 0x00000002,
    /** Image */
    VNCVisualContentCategoryImage          = 0x00000004,
    /** Vector graphics */
    VNCVisualContentCategoryGraphicsVector = 0x00000080,
    /** 3D graphics */
    VNCVisualContentCategoryGraphics3D     = 0x00000010,
    /** User interface (e.g. application menu) */
    VNCVisualContentCategoryUI             = 0x00000020,
    /** Car mode */
    VNCVisualContentCategoryCarMode        = 0x00010000,
    /** Miscellaneous */
    VNCVisualContentCategoryMisc           = 0x80000000,

} VNCVisualContentCategory;

/**
 * \brief Constants used to define MirrorLink audio content categories.
 *
 * This field is a bit mask that specifies the type or types of audio content
 * that an application will provide.
 *
 * For further information, see Appendix A, 'UPnP Application Server Service'
 * in the MirrorLink specification.
 */
typedef enum
{
    /** Phone audio. */
    VNCAudioContentCategoryPhone            = 0x00000001,
    /** Media audio out. */
    VNCAudioContentCategoryMediaOut         = 0x00000002,
    /** Media audio in. */
    VNCAudioContentCategoryMediaIn          = 0x00000004,
    /** Voice command out. */
    VNCAudioContentCategoryVoiceCommandOut  = 0x00000008,
    /** Voice command in. */
    VNCAudioContentCategoryVoiceCommandIn   = 0x00000010,
    /** Miscellaneous content. */
    VNCAudioContentCategoryMisc             = 0x80000000,

} VNCAudioContentCategory;

/**
 * \brief Describes the level of trust that the MirrorLink server has in the
 * Context Information that it is providing.
 *
 * For further information, see Appendix A, 'UPnP Application Server Service',
 * in the MirrorLink specification.
 *
 * \if VIEWER_SDK
 * \see VNCContextInformation, VNCViewerContextInformationCallback
 * \endif
 */
typedef enum
{
    /** The server has no trust in the reported information. */
    VNCTrustLevelUnknown                   = 0x00,
    /** The provided data is under the control of the user. */
    VNCTrustLevelUserConfiguration         = 0x40,
    /** The provided data is under the control of the application. */
    VNCTrustLevelSelfRegisteredApplication = 0x60,
    /**
     * The provided data is under sole control of the VNC and UPnP server.
     * The application is known to them and has been uniquely identified.
     */
    VNCTrustLevelRegisteredApplication     = 0x80,
    /**
     * The provided data is under sole control of the VNC and UPnP server.
     * The data is derived from a valid application certificate.
     */
    VNCTrustLevelApplicationCertificate    = 0xA0

} VNCTrustLevel;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the knobKeySupport field in the 
 * VNCServerEventConfiguration and \ref VNCClientEventConfiguration structures.
 * \else
 * \brief Constants expressing knob shift and rotate configuration settings.
 * \endif
 *
 * \if VIEWER_SDK
 * In the VNCServerEventConfiguration structure, these values are combined to
 * indicate to the viewer application which input events the server supports.
 *
 * In the \ref VNCClientEventConfiguration structure, these values are combined
 * to indicate to the server which events the viewer application will generate.
 *
 * Further details about what the shifting, pushing, pulling and rotating of
 * different knobs mean can be found in mirrorlinkkeysymdef.h.
 * \else
 * These values are combined to indicate the knob events that are supported by
 * a MirrorLink server or client.
 * \endif
 *
 * For further information, see Appendix A, 'Knob Configuration' in the
 * MirrorLink VNC based Display and Control specification.
 *
 * \if VIEWER_SDK
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 * \endif
 */
typedef enum
{
    /** \cond */
    VNCKnobKeySupport0ShiftX  = 0x00000001,
    VNCKnobKeySupport0ShiftY  = 0x00000002,
    VNCKnobKeySupport0ShiftXY = 0x00000004,
    VNCKnobKeySupport0PushZ   = 0x00000008,
    VNCKnobKeySupport0PullZ   = 0x00000010,
    VNCKnobKeySupport0RotateX = 0x00000020,
    VNCKnobKeySupport0RotateY = 0x00000040,
    VNCKnobKeySupport0RotateZ = 0x00000080,
    VNCKnobKeySupport1ShiftX  = 0x00000100,
    VNCKnobKeySupport1ShiftY  = 0x00000200,
    VNCKnobKeySupport1ShiftXY = 0x00000400,
    VNCKnobKeySupport1PushZ   = 0x00000800,
    VNCKnobKeySupport1PullZ   = 0x00001000,
    VNCKnobKeySupport1RotateX = 0x00002000,
    VNCKnobKeySupport1RotateY = 0x00004000,
    VNCKnobKeySupport1RotateZ = 0x00008000,
    VNCKnobKeySupport2ShiftX  = 0x00010000,
    VNCKnobKeySupport2ShiftY  = 0x00020000,
    VNCKnobKeySupport2ShiftXY = 0x00040000,
    VNCKnobKeySupport2PushZ   = 0x00080000,
    VNCKnobKeySupport2PullZ   = 0x00100000,
    VNCKnobKeySupport2RotateX = 0x00200000,
    VNCKnobKeySupport2RotateY = 0x00400000,
    VNCKnobKeySupport2RotateZ = 0x00800000,
    VNCKnobKeySupport3ShiftX  = 0x01000000,
    VNCKnobKeySupport3ShiftY  = 0x02000000,
    VNCKnobKeySupport3ShiftXY = 0x04000000,
    VNCKnobKeySupport3PushZ   = 0x08000000,
    VNCKnobKeySupport3PullZ   = 0x10000000,
    VNCKnobKeySupport3RotateX = 0x20000000,
    VNCKnobKeySupport3RotateY = 0x40000000,
    VNCKnobKeySupport3RotateZ = 0x80000000,
    /** \endcond */
} VNCKnobKeySupport;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the deviceKeySupport field in the 
 * VNCServerEventConfiguration and \ref VNCClientEventConfiguration structures.
 * \else
 * \brief Constants expressing device keys that may be supported.
 * \endif
 *
 * \if VIEWER_SDK
 * In the VNCServerEventConfiguration structure, these values are combined to
 * indicate to the viewer application which input events the server supports.
 *
 * In the \ref VNCClientEventConfiguration structure, these values are combined
 * to indicate to the server which events the viewer application will generate.
 *
 * Further details on the meaning of these keys can be found in
 * mirrorlinkkeysymdef.h.
 * \else
 * These values may be combined to indicate the device keys that a MirrorLink
 * server or client supports.
 * \endif
 *
 * \if VIEWER_SDK
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 * \endif
 */
typedef enum
{
    /** \cond */
    VNCDeviceKeySupportPhoneCall   = 0x00000001,
    VNCDeviceKeySupportPhoneEnd    = 0x00000002,
    VNCDeviceKeySupportSoftLeft    = 0x00000004,
    VNCDeviceKeySupportSoftMiddle  = 0x00000008,
    VNCDeviceKeySupportSoftRight   = 0x00000010,
    VNCDeviceKeySupportApplication = 0x00000020,
    VNCDeviceKeySupportOk          = 0x00000040,
    VNCDeviceKeySupportDelete      = 0x00000080,
    VNCDeviceKeySupportZoomIn      = 0x00000100,
    VNCDeviceKeySupportZoomOut     = 0x00000200,
    VNCDeviceKeySupportClear       = 0x00000400,
    VNCDeviceKeySupportForward     = 0x00000800,
    VNCDeviceKeySupportBackward    = 0x00001000,
    VNCDeviceKeySupportHome        = 0x00002000,
    VNCDeviceKeySupportSearch      = 0x00004000,
    VNCDeviceKeySupportMenu        = 0x00008000,
    VNCDeviceKeySupportALL         = 0x0000ffff,
    /** \endcond */
} VNCDeviceKeySupport;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the multimediaKeySupport field in the 
 * VNCServerEventConfiguration and \ref VNCClientEventConfiguration structures.
 * \else
 * \brief Constants expressing multimedia keys that may be supported.
 * \endif
 *
 * \if VIEWER_SDK
 * In the VNCServerEventConfiguration structure, these values are combined to
 * indicate to the viewer application which input events the server supports.
 *
 * In the \ref VNCClientEventConfiguration structure, these values are combined
 * to indicate to the server which events the viewer application will generate.
 *
 * Further details on the meaning of these keys can be found in
 * mirrorlinkkeysymdef.h.
 * \else
 * These values may be combined to indicate the multimedia keys that a
 * MirrorLink server or client supports.
 * \endif
 *
 * \if VIEWER_SDK
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 * \endif
 */
typedef enum
{
    /** \cond */
    VNCMultimediaKeySupportPlay     = 0x00000001,
    VNCMultimediaKeySupportPause    = 0x00000002,
    VNCMultimediaKeySupportStop     = 0x00000004,
    VNCMultimediaKeySupportForward  = 0x00000008,
    VNCMultimediaKeySupportRewind   = 0x00000010,
    VNCMultimediaKeySupportNext     = 0x00000020,
    VNCMultimediaKeySupportPrevious = 0x00000040,
    VNCMultimediaKeySupportMute     = 0x00000080,
    VNCMultimediaKeySupportUnmute   = 0x00000100,
    VNCMultimediaKeySupportPhoto    = 0x00000200,
    /** \endcond */
} VNCMultimediaKeySupport;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the features field in the
 * \ref VNCDeviceStatusRequest and VNCDeviceStatus structures.
 * \else
 * \brief Constants for use with the features field in the
 * \ref VNCDeviceStatus structure.
 * \endif
 *
 * \if VIEWER_SDK
 * In the \ref VNCDeviceStatusRequest structure, these values are combined to
 * indicate to the server which feature or features the viewer applications
 * wishes to use at this time.  Use 'Ignored' to indicate that you do not wish
 * to alter the state of a feature.
 *
 * In the VNCDeviceStatus structure, these values are combined to indicate to
 * the viewer application the current status of one or more features.
 * 'Ignored' indicates that the state of the feature is not given in this
 * message (usually because 'Ignored' was also given in the corresponding
 * DeviceStatusRequest).
 * \else
 * These values are combined to indicate the current status of one or more
 * features. 'Ignored' indicates that the state of the feature is not given in
 * this message.
 * \endif
 *
 * Use the MASK enum members to mask the relevant features out for testing:
 *
 * \code
 * VNCDeviceStatus deviceStatus;
 * ...
 * if ((deviceStatus.features & VNCDeviceStatusFeatureOrientationMASK) ==
 *     VNCDeviceStatusFeatureOrientationLandscape)
 * {
 *     // The server display is currently in landscape orientation.
 * }
 * \endcode
 *
 * \if VIEWER_SDK
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 * \endif
 */
typedef enum
{
    /** This message does not specify the status of the key lock. */
    VNCDeviceStatusFeatureKeyLockIgnored                = 0x00000000,
    /**
     * Key lock is or should be disabled.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureKeyLockIgnored.
     */
    VNCDeviceStatusFeatureKeyLockDisabled               = 0x00000002,
    /**
     * Key lock is or should be enabled.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureKeyLockIgnored.
     */
    VNCDeviceStatusFeatureKeyLockEnabled                = 0x00000003,
    /** Used to mask out the key lock status for testing. */
    VNCDeviceStatusFeatureKeyLockMASK                   = 0x00000003,

    /** This message does not specify the status of the device lock. */
    VNCDeviceStatusFeatureDeviceLockIgnored             = 0x00000000,
    /**
     * Device lock is or should be disabled.
     *
     * This feature has been deprecated in MirrorLink 1.3 for clients. Viewer
     * applications should always use VNCDeviceStatusFeatureDeviceLockIgnored
     * in device status requests if they support MirrorLink 1.3 or above.
     * Server applications which support MirrorLink 1.3 or above may still
     * report the current device lock status.
     */
    VNCDeviceStatusFeatureDeviceLockDisabled            = 0x00000008,
    /**
     * Device lock is or should be enabled.
     *
     * This feature has been deprecated in MirrorLink 1.3 for clients. Viewer
     * applications should always use VNCDeviceStatusFeatureDeviceLockIgnored
     * in device status requests if they support MirrorLink 1.3 or above.
     * Server applications which support MirrorLink 1.3 or above may still
     * report the current device lock status.
     */
    VNCDeviceStatusFeatureDeviceLockEnabled             = 0x0000000C,
    /** Used to mask out the device lock status for testing. */
    VNCDeviceStatusFeatureDeviceLockMASK                = 0x0000000C,

    /** This message does not specify the status of the screensaver. */
    VNCDeviceStatusFeatureScreenSaverIgnored            = 0x00000000,
    /**
     * Screensaver is or should be disabled.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureScreenSaverIgnored.
     */
    VNCDeviceStatusFeatureScreenSaverDisabled           = 0x00000020,
    /**
     * Screensaver is or should be enabled.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureScreenSaverIgnored.
     */
    VNCDeviceStatusFeatureScreenSaverEnabled            = 0x00000030,
    /** Used to mask out the screensaver status for testing. */
    VNCDeviceStatusFeatureScreenSaverMASK               = 0x00000030,

    /** This message does not specify the status of night mode. */
    VNCDeviceStatusFeatureNightModeIgnored              = 0x00000000,
    /** Night mode is or should be disabled. */
    VNCDeviceStatusFeatureNightModeDisabled             = 0x00000080,
    /** Night mode is or should be enabled. */
    VNCDeviceStatusFeatureNightModeEnabled              = 0x000000C0,
    /** Used to mask out the night mode status for testing. */
    VNCDeviceStatusFeatureNightModeMASK                 = 0x000000C0,

    /** This message does not specify the status of voice input. */
    VNCDeviceStatusFeatureVoiceInputIgnored             = 0x00000000,
    /** Voice input is or should be disabled. */
    VNCDeviceStatusFeatureVoiceInputDisabled            = 0x00000200,
    /** Voice input is or should be enabled. */
    VNCDeviceStatusFeatureVoiceInputEnabled             = 0x00000300,
    /** Used to mask out the voice input status for testing. */
    VNCDeviceStatusFeatureVoiceInputMASK                = 0x00000300,

    /** This message does not specify the status of microphone input. */
    VNCDeviceStatusFeatureMicrophoneInputIgnored        = 0x00000000,
    /** Microphone input is or should be disabled. */
    VNCDeviceStatusFeatureMicrophoneInputDisabled       = 0x00000800,
    /** Microphone input is or should be enabled. */
    VNCDeviceStatusFeatureMicrophoneInputEnabled        = 0x00000C00,
    /** Used to mask out the microphone input status for testing. */
    VNCDeviceStatusFeatureMicrophoneInputMASK           = 0x00000C00,

    /** \deprecated - Use VNCDeviceStatusFeatureMicrophoneInputIgnored instead. */
    VNCDeviceStatusFeatureVoiceInputReroutingIgnored    = VNCDeviceStatusFeatureMicrophoneInputIgnored,
    /** \deprecated - Use VNCDeviceStatusFeatureMicrophoneInputDisabled instead. */
    VNCDeviceStatusFeatureVoiceInputReroutingDisabled   = VNCDeviceStatusFeatureMicrophoneInputDisabled,
    /** \deprecated - Use VNCDeviceStatusFeatureMicrophoneInputEnabled instead. */
    VNCDeviceStatusFeatureVoiceInputReroutingEnabled    = VNCDeviceStatusFeatureMicrophoneInputEnabled,
    /** \deprecated - Use VNCDeviceStatusFeatureMicrophoneInputMASK instead. */
    VNCDeviceStatusFeatureVoiceInputReroutingMASK       = VNCDeviceStatusFeatureMicrophoneInputMASK,

    /** This message does not specify the status of driver distraction
     * avoidance. */
    VNCDeviceStatusFeatureDriverDistractionAvoidanceIgnored  = 0x00000000,
    /** Driver distraction avoidance is or should be disabled. */
    VNCDeviceStatusFeatureDriverDistractionAvoidanceDisabled = 0x00020000,
    /** Driver distraction avoidance is or should be enabled. */
    VNCDeviceStatusFeatureDriverDistractionAvoidanceEnabled  = 0x00030000,
    /** Used to mask out the driver distraction avoidance status for
     * testing. */
    VNCDeviceStatusFeatureDriverDistractionAvoidanceMASK     = 0x00030000,

    /** This message does not specify the framebuffer rotation. */
    VNCDeviceStatusFeatureFramebufferRotationIgnored    = 0x00000000,
    /** The server framebuffer is not or should not be rotated. */
    VNCDeviceStatusFeatureFramebufferRotation0Degrees   = 0x04000000,
    /**
     * The server framebuffer is or should be rotated 90 degrees clockwise.
     *
     * This feature has been deprecated in MirrorLink 1.3. Server applications
     * should only use VNCDeviceStatusFeatureFramebufferRotation0Degrees if
     * they support MirrorLink 1.3 or above. Viewer applications which support
     * MirrorLink 1.3 or above may use either
     * VNCDeviceStatusFeatureFramebufferRotation0Degrees or
     * VNCDeviceStatusFeatureFramebufferRotationIgnored.
     */
    VNCDeviceStatusFeatureFramebufferRotation90Degrees  = 0x05000000,
    /**
     * The server framebuffer is or should be rotated 180 degrees.
     *
     * This feature has been deprecated in MirrorLink 1.3. Server applications
     * should only use VNCDeviceStatusFeatureFramebufferRotation0Degrees if
     * they support MirrorLink 1.3 or above. Viewer applications which support
     * MirrorLink 1.3 or above may use either
     * VNCDeviceStatusFeatureFramebufferRotation0Degrees or
     * VNCDeviceStatusFeatureFramebufferRotationIgnored.
     */
    VNCDeviceStatusFeatureFramebufferRotation180Degrees = 0x06000000,
    /**
     * The server framebuffer is or should be rotated 270 degrees clockwise
     * (or, equivalently, 90 degrees counter-clockwise).
     *
     * This feature has been deprecated in MirrorLink 1.3. Server applications
     * should only use VNCDeviceStatusFeatureFramebufferRotation0Degrees if
     * they support MirrorLink 1.3 or above. Viewer applications which support
     * MirrorLink 1.3 or above may use either
     * VNCDeviceStatusFeatureFramebufferRotation0Degrees or
     * VNCDeviceStatusFeatureFramebufferRotationIgnored.
     */
    VNCDeviceStatusFeatureFramebufferRotation270Degrees = 0x07000000,
    /** Used to mask out the framebuffer rotation for testing. */
    VNCDeviceStatusFeatureFramebufferRotationMASK       = 0x07000000,

    /**
     * This message does not specify the display orientation.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureOrientationLandscape.
     */
    VNCDeviceStatusFeatureOrientationIgnored            = 0x00000000,
    /** The display orientation is or should be landscape. */
    VNCDeviceStatusFeatureOrientationLandscape          = 0x10000000,
    /**
     * The display orientation is or should be portrait.
     *
     * This feature has been deprecated in MirrorLink 1.3. If your application
     * supports MirrorLink 1.3 or above, then you should only use
     * VNCDeviceStatusFeatureOrientationLandscape.
     */
    VNCDeviceStatusFeatureOrientationPortrait           = 0x18000000,
    /** Used to mask out the display orientation for testing. */
    VNCDeviceStatusFeatureOrientationMASK               = 0x18000000,
} VNCDeviceStatusFeature;

/**
 * \brief Structure holding device status information.
 *
 * \if VIEWER_SDK
 * This holds a decoded DeviceStatus MirrorLink extension message that has been
 * received from the server. Each received DeviceStatus message is decoded into
 * a VNCDeviceStatus structure and passed to the optional
 * VNCViewerDeviceStatusCallback.
 * \else
 * This structure may be used to send or receive device status information.
 * \endif
 *
 * \if VIEWER_SDK
 * \see VNCViewerDeviceStatusCallback,
 * VNCViewerSendDeviceStatusRequest, VNCDeviceStatusRequest,
 * VNCDeviceStatusFeature
 * \endif
 */
typedef struct
{
    /**
     * A combination of values from the \ref VNCDeviceStatusFeature enumeration.
     *
     * Some features have been deprecated in MirrorLink 1.3. Your application
     * should not attempt to use deprecated feature status values if it
     * supports MirrorLink 1.3 or above. See \ref VNCDeviceStatusFeature for
     * further information.
     */
    vnc_uint32_t features;
} VNCDeviceStatus;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the blockReason field of the
 * VNCFramebufferBlockingNotification structure.
 * \else
 * \brief Constants expressing reasons why a framebuffer has been completely
 * or partially blocked.
 * \endif
 *
 * \if VIEWER_SDK
 * Use a combination of these values to inform the server why the viewer
 * application has chosen to block an application's visual content.  For
 * example, the viewer application may have decided (based on the application's
 * category and content category) that the application does not comply with the
 * current driver distraction policy, or the viewer application may not be
 * visible to the end user at this time.
 * \else
 * These values may be combined to indicate multiple reasons for the blocking
 * of visual content.
 * \endif
 *
 * \if VIEWER_SDK
 * \see VNCFramebufferBlockingNotification,
 * VNCViewerSendFramebufferBlockingNotification,
 * VNCContextInformation, VNCViewerContextInformationCallback
 * \endif
 */
typedef enum
{
    /**
     * The application's content category has been disallowed (e.g. by the
     * driver distraction policy).
     *
     * This reason has been deprecated in MirrorLink 1.1 and is forbidden to be
     * used in MirrorLink 1.3. Viewer applications should not specify this
     * reason if they support MirrorLink 1.3 or above. Server applications
     * supporting MirrorLink 1.3 or above should reinterpret this as
     * VNCFramebufferBlockReasonApplicationNotTrusted.
     */
    VNCFramebufferBlockReasonContentCategoryNotAllowed     = 0x0001,
    /**
     * The application category has been disallowed (e.g. by the driver
     * distraction policy).
     *
     * This reason has been deprecated and is forbidden to be
     * used in MirrorLink 1.3. Viewer applications should not specify this
     * reason if they support MirrorLink 1.3 or above. Server applications
     * supporting MirrorLink 1.3 or above should reinterpret this as
     * VNCFramebufferBlockReasonApplicationNotTrusted.
     */
    VNCFramebufferBlockReasonApplicationCategoryNotAllowed = 0x0002,
    /**
     * The server's trust in the content category that it reported is not
     * sufficient to satisfy the viewer application.
     *
     * This reason has been deprecated in MirrorLink 1.1 and is forbidden to be
     * used in MirrorLink 1.3. Viewer applications should not specify this
     * reason if they support MirrorLink 1.3 or above. Server applications
     * supporting MirrorLink 1.3 or above should reinterpret this as
     * VNCFramebufferBlockReasonApplicationNotTrusted.
     */
    VNCFramebufferBlockReasonContentNotTrusted             = 0x0004,
    /**
     * The server's trust in the application category that it reported is not
     * sufficient to satisfy the viewer application.
     */
    VNCFramebufferBlockReasonApplicationNotTrusted         = 0x0008,
    /**
     * The server application has not followed the content rules that were
     * communicated to the server via UPnP.
     *
     * This reason has been deprecated in MirrorLink 1.1 and is forbidden to be
     * used in MirrorLink 1.3. Viewer applications should not specify this
     * reason if they support MirrorLink 1.3 or above. Server applications
     * supporting MirrorLink 1.3 or above should reinterpret this as
     * VNCFramebufferBlockReasonApplicationNotTrusted.
     */
    VNCFramebufferBlockReasonContentRulesNotFollowed       = 0x0010,
    /**
     * The server application has been disallowed based on its unique ID (e.g.
     * by the driver distraction policy).
     *
     * This reason has been deprecated and is forbidden to be used in
     * MirrorLink 1.3. Viewer applications should not specify this reason if
     * they support MirrorLink 1.3 or above. Server applications supporting
     * MirrorLink 1.3 or above should reinterpret this as
     * VNCFramebufferBlockReasonApplicationNotTrusted.
     */
    VNCFramebufferBlockReasonApplicationUniqueIdNotAllowed = 0x0020,
    /**
     * The viewer application is not in focus.
     *
     * This reason has been deprecated in MirrorLink 1.1 and is forbidden to be
     * used in MirrorLink 1.3. Viewer applications should not specify this
     * reason if they support MirrorLink 1.3 or above.
     */
    VNCFramebufferBlockReasonUINotInFocus                  = 0x0100,
    /**
     * The UI of the viewer application is not visible to the user.
     */
    VNCFramebufferBlockReasonUINotVisible                  = 0x0200,
    /**
     * UI layout not supported.
     */
    VNCFramebufferBlockReasonUILayoutNotSupported          = 0x0400,
} VNCFramebufferBlockReason;

/**
 * \if VIEWER_SDK
 * \brief Constants for use with the blockReason field of the
 * VNCAudioBlockingNotification structure.
 * \else
 * \brief Constants expressing reasons why audio content has been completely or
 * partially blocked.
 * \endif
 *
 * \if VIEWER_SDK
 * Use a combination of these values to inform the server why the viewer
 * application has chosen to block the server's audio content.
 *
 * VNCAudioBlockReasonUnblock may be used instead to inform the server that the
 * audio content must be resumed.
 * \else
 * These values may be combined to indicate multiple reasons for the blocking
 * of audio content.
 *
 * VNCAudioBlockReasonUnblock may be used instead to indicate that the audio
 * content must be resumed.
 * \endif
 *
 * \if VIEWER_SDK
 * \see VNCAudioBlockingNotification,
 * VNCViewerSendAudioBlockingNotification
 * \endif
 */
typedef enum
{
    /**
     * The application's blocked audio stream should be resumed.
     */
    VNCAudioBlockReasonUnblock                       = 0x0000,
    /**
     * The application's category has been disallowed (e.g. by the driver
     * distraction policy).
     *
     * This reason has been deprecated in MirrorLink 1.3. Viewer applications
     * should not specify this reason if they support MirrorLink 1.3 or above.
     * Server applications supporting MirrorLink 1.3 or above should
     * reinterpret this as VNCAudioBlockReasonStreamMuted.
     */
    VNCAudioBlockReasonApplicationCategoryNotAllowed = 0x0001,
    /**
     * The server's trust in the application category that it reported is not
     * sufficient to satisfy the viewer application.
     *
     * This reason has been deprecated in MirrorLink 1.3. Viewer applications
     * should not specify this reason if they support MirrorLink 1.3 or above.
     * Server applications supporting MirrorLink 1.3 or above should
     * reinterpret this as VNCAudioBlockReasonStreamMuted.
     */
    VNCAudioBlockReasonApplicationNotTrusted         = 0x0002,
    /**
     * The server application has been disallowed based on its unique ID.
     *
     * This reason has been deprecated in MirrorLink 1.3. Viewer applications
     * should not specify this reason if they support MirrorLink 1.3 or above.
     * Server applications supporting MirrorLink 1.3 or above should
     * reinterpret this as VNCAudioBlockReasonStreamMuted.
     */
    VNCAudioBlockReasonApplicationUniqueIdNotAllowed = 0x0004,
    /**
     * The user has muted all audio.
     */
    VNCAudioBlockReasonGloballyMuted                 = 0x0008,
    /**
     * The user has muted a particular audio stream.
     */
    VNCAudioBlockReasonStreamMuted                   = 0x0010,
} VNCAudioBlockReason;

#ifdef __cplusplus
}
#endif

#endif    // UUID_2cee7b5a_42ec_11e3_a170_001cc047a813
